library(xlsx)
library(ggplot2)
library(ape)
library(nlme)

data <- read.xlsx("Bird_CoT.xlsx", sheetIndex = 1)
tree <- read.tree("Bird_Tree.tre")

data.M1 <- data[c(1:4,6:9,13:16),] #Model 1 - no ducks or broilers
data.M2 <- data[c(1:4,6:16),] #All birds except broilers
data.M3 <- data[c(1:3,6:10,13:16),] #phylogenetic, only 1 breed per species

row.names(data.M3) <- data.M3$Taxon
data.M3 <- data.M3[match(tree$tip.label,rownames(data.M3)),] #matches row order to tips

model1 <- lm(Ln_COT ~ Ln_BM, data.M1)
summary(model1)

model2 <- lm(Ln_COT ~ Ln_BM, data.M2)
summary(model2)

model3 <- gls(Ln_COT ~ Ln_BM,
                 correlation = corPagel(1, phy = tree),
                 data = data.M3,
                 method = "ML")
summary(model3)

xmin <- min(data.M3$Ln_BM)
xmax <- max(data.M3$Ln_BM)
ymin <- model3$coefficients[[1]] + model3$coefficients[[2]] * xmin
ymax <- model3$coefficients[[1]] + model3$coefficients[[2]] * xmax

tiff(filename = "Min_CoT.tif", compression = "lzw", width = 222, height = 166, units = "mm", res = 600) #166x166*2 or 166x166*1.5

ggplot(data.M1, aes(x=Ln_BM, y=Ln_COT)) +
  geom_smooth(aes(color = "black"), linetype = 1, method = "lm", se = TRUE, size = 0.8) +
  geom_segment(aes(x = xmin, xend = xmax, y = ymin, yend = ymax), color = "black", linetype = "dotted", size = 0.8) +
  geom_point(aes(fill = "black"), size = 2.5, shape = 24) +
  geom_point(data = data[5,], aes(x=Ln_BM, y=Ln_COT, fill = "red"), size = 2.5, shape = 25) +
  geom_point(data = data[10,], aes(x=Ln_BM, y=Ln_COT, fill = "lightyellow3"), size = 3, shape = 22) +
  geom_point(data = data[11,], aes(x=Ln_BM, y=Ln_COT, fill = "sienna3"), size = 3, shape = 21) +
  geom_point(data = data[12,], aes(x=Ln_BM, y=Ln_COT, fill = "skyblue4"), size = 3, shape = 23) +
  theme(panel.grid.major = element_blank(), 
        panel.grid.minor = element_blank(), 
        panel.background = element_blank(), 
        axis.line = element_line(colour = "black")) +
  theme(legend.position="none") +
  xlab("Log Body Mass (kg)") + 
  #ylab(expression(paste("Log CoT (J kg"^"-1"*"m"^"-1"*")"))) +
  #scale_shape_manual(values = c(21, 1, 5)) +
  scale_color_manual(values = "black") +
  scale_fill_manual(values = c('black', 
                                'lightyellow3', 
                                'red', 
                                'sienna3',
                                'skyblue4'), guide = 'none') +
  annotate(geom="text", x=1, y=2, label="y = 1.391x -0.456 ",color="black")

dev.off()
